import React from 'react';
import { Link, useNavigate, useLocation } from 'react-router-dom';
import { useTheme, ThemeVariant } from '@/context/ThemeContext'; // Removed ThemeMode as it's part of currentMode string
import { useAuth } from '@/context/AuthContext';
import SidebarNav from './SidebarNav'; // ADDED import

// Placeholder for navigation items - this might become dynamic or configurable
const navItems = [
  { name: "Home", path: "/home" }, // MODIFIED path to /home
  { name: "About", path: "/about" }, // Example future path
  { name: "Videos", path: "/videos" }, // ADDED Videos link
  { name: "Contact", path: "/contact" }, // Example future path
];

const HomePageLayout: React.FC = () => {
  const { currentVariant, currentMode, setCurrentVariant, toggleMode } = useTheme();
  const { currentUser, logout } = useAuth();
  const navigate = useNavigate();
  const location = useLocation();

  console.log('[HomePageLayout] Current Variant:', currentVariant);
  console.log('[HomePageLayout] Current Mode:', currentMode);
  console.log('[HomePageLayout] Current Location:', location.pathname); // Optional: for debugging

  // The main background image and other theme-specific assets 
  // will now be primarily controlled by CSS based on the theme-variant and mode classes.
  // We might still need a way to pass specific image URLs if they can't be purely CSS.

  return (
    <div
      className={`theme-page-layout-container theme-variant-${currentVariant} mode-${currentMode}`}
      // style={{ backgroundImage: `url(...)` }} // Controlled by CSS now
    >
      <header className="theme-header">
        {/* Optional: A header image, if it's common or also CSS driven */}
        {/* <img src="/path/to/generic-header-image.png" alt="Header" className="theme-header-image" /> */}
        
        <div className="theme-header-overlay">
          <h1 className="theme-site-title">CODA Project</h1> {/* MODIFIED Title */}
          
          {/* This div will be styled by App2.css to float in top right */}
          <div className="theme-page-controls-container">
            <div className="auth-controls-themed-page"> {/* Vertical stack item 1: Auth */}
              {currentUser ? (
                <button onClick={async () => {
                  try {
                    await logout();
                  } catch (error) {
                    console.error("Failed to log out", error);
                  }
                }} className="auth-button logout-button">
                  Logout ({currentUser.email?.substring(0, currentUser.email.indexOf('@')) || 'User'})
                </button>
              ) : (
                <>
                  <Link to="/login" className="auth-button login-button">Login</Link>
                  <Link to="/signup" className="auth-button signup-button">Sign Up</Link>
                </>
              )}
            </div>

            <div className="theme-selection-group"> {/* Vertical stack item 2: Toggle + Dropdown */}
              <div className="theme-toggle">
                <button 
                  onClick={toggleMode}
                  className={`toggle-switch ${currentMode}`}
                  aria-label={`Switch to ${currentMode === 'light' ? 'dusky' : 'light'} mode`}
                >
                  <span className="toggle-slider"></span>
                  <span className="toggle-icon">{currentMode === 'light' ? '☀️' : '🌙'}</span>
                </button>
              </div>
              <select 
                value={currentVariant} 
                onChange={(e) => {
                  const newVariant = e.target.value as ThemeVariant;
                  setCurrentVariant(newVariant);
                  if (location.pathname !== '/home') {
                    const pathVariant = newVariant.toLowerCase().replace(/\s+/g, ''); 
                    navigate(`/${pathVariant}`);
                  }
                }} 
                className="theme-select"
                aria-label="Select theme variant"
              >
                <option value="floral">🌸 Floral</option>
                <option value="simple">🎯 Simple</option>
                <option value="cabin">🏡 Cabin</option>
                <option value="nightSky">✨ Sunset Dream</option>
              </select>
            </div>
          </div>

          <nav className="theme-navigation">
            {navItems.map(item => (
              <Link key={item.name} to={item.path} className="theme-nav-link">{item.name}</Link>
            ))}
          </nav>
        </div>
      </header>

      <div className="theme-body-wrapper">
        <div className="theme-content-area"> {/* This was .floral-content */}
          <aside className="theme-sidebar">
            <p><strong>Navigation</strong></p>
            <SidebarNav /> {/* REPLACED with SidebarNav component */}
          </aside>

          <main className="theme-main-content"> {/* This was .floral-main */}
            {location.pathname === '/videos' ? (
              <div className="videos-page-navigation">
                <Link to="/home" className="theme-nav-link videos-nav-button">Home</Link>
                <Link to="/videos/guided" className="theme-nav-link videos-nav-button">Guided</Link>
                <Link to="/videomap" className="theme-nav-link videos-nav-button">Video Map</Link>
                <button onClick={() => navigate(-1)} className="theme-nav-link videos-nav-button">Back</button>
              </div>
            ) : (
              <>
                <h2>Welcome!</h2>
                <p>This is the main content area. It will be styled based on the active theme.</p>
                {/* Example of how content might be structured */}
                <article>
                  <h3>Article Title</h3>
                  <p>Some text here...</p>
                </article>
                <article>
                  <h3>Another Article</h3>
                  <p>More text here...</p>
                </article>
              </>
            )}
          </main>
        </div>
      </div>
       {/* Optional: A common footer */}
       {/* <footer className="theme-footer"> <p>&copy; 2024 My Application</p> </footer> */}
    </div>
  );
};

export default HomePageLayout; 